import { useEffect, useRef, useState } from "react";
import { gsap, ScrollTrigger } from "@/helpers/gsap";

export default function HorizontalGalleryGSAP() {
  const containerRef = useRef<HTMLDivElement | null>(null);
  const itemsRef = useRef<HTMLUListElement | null>(null);
  const tlRef = useRef<gsap.core.Timeline | null>(null);

  // Synchronized Scroll Animation Refs
  const scrollVelocityRef = useRef(0);
  const lastScrollTopRef = useRef(0);
  const scrollEndTimerRef = useRef<NodeJS.Timeout | null>(null);
  const tickingRef = useRef(false);
  const touchTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const currentScaleRef = useRef(1);
  const isScrollingRef = useRef(false);

  // Synchronized Scroll Configuration (moved outside useEffect to avoid dependency issues)
  const syncScrollConfig = useRef({
    cardSelector: "li", // Target all li elements
    minScale: 0.9,
    maxVelocity: 50,
    scrollEndDelay: 100,
    scalingTransition: "transform 0.3s ease-out, opacity 0.1s ease-out",
    bounceTransition: "transform 0.6s, opacity 0.6s",
    // minOpacity: 0.85
  });

  useEffect(() => {
    // GSAP Horizontal Gallery Logic
    const build = () => {
      const container = containerRef.current;
      const items = itemsRef.current;
      if (!container || !items) return;

      const totalWidth = items.scrollWidth;
      const startOffset = items.offsetLeft;
      const distance = totalWidth + startOffset;

      if (tlRef.current) {
        tlRef.current.scrollTrigger?.kill();
        tlRef.current.kill();
        tlRef.current = null;
      }

      tlRef.current = gsap
        .timeline({
          ease: "none",
          scrollTrigger: {
            trigger: container,
            pin: true,
            end: () => "+=" + distance,
            scrub: 1,
            invalidateOnRefresh: true,
          },
        })
        // sirf horizontal travel
        .to(items, { x: -distance, ease: "none" });
    };

    // Synchronized Scroll Animation Logic
    const updateScrollPosition = () => {
      const scrollTop = window.pageYOffset;

      // Calculate scroll velocity
      scrollVelocityRef.current = Math.abs(
        scrollTop - lastScrollTopRef.current
      );
      lastScrollTopRef.current = scrollTop;

      isScrollingRef.current = true;
      requestAnimationFrame(updateCards);
    };

    const updateCards = () => {
      const config = syncScrollConfig.current;
      const cards = document.querySelectorAll(
        `${containerRef.current ? ".review-cards " : ""}${config.cardSelector}`
      );

      if (isScrollingRef.current && scrollVelocityRef.current > 0) {
        // Calculate scale based on scroll velocity
        const velocityRatio = Math.min(
          scrollVelocityRef.current / config.maxVelocity,
          1
        );

        // Scale all cards simultaneously based on scroll speed
        currentScaleRef.current = 1 - velocityRatio * (1 - config.minScale);
        currentScaleRef.current = Math.max(
          config.minScale,
          Math.min(1, currentScaleRef.current)
        );

        // Apply same scale and opacity to all cards
        cards.forEach((card: Element) => {
          const htmlCard = card as HTMLElement;
          htmlCard.style.transition = config.scalingTransition;
          htmlCard.style.transform = `scale(${currentScaleRef.current})`;
          // htmlCard.style.opacity = Math.max(config.minOpacity, currentScaleRef.current).toString();
        });
      }

      tickingRef.current = false;
    };

    const onScroll = () => {
      if (!tickingRef.current) {
        requestAnimationFrame(updateScrollPosition);
        tickingRef.current = true;
      }

      // Clear previous timer
      if (scrollEndTimerRef.current) clearTimeout(scrollEndTimerRef.current);

      // Set new timer for scroll end
      scrollEndTimerRef.current = setTimeout(() => {
        isScrollingRef.current = false;
        scrollVelocityRef.current = 0;
        bounceBackCards();
      }, syncScrollConfig.current.scrollEndDelay);
    };

    const bounceBackCards = () => {
      const config = syncScrollConfig.current;
      const cards = document.querySelectorAll(
        `${containerRef.current ? ".review-cards " : ""}${config.cardSelector}`
      );

      // Bounce all cards back to full size simultaneously
      cards.forEach((card: Element) => {
        const htmlCard = card as HTMLElement;
        htmlCard.style.transition = config.bounceTransition;
        htmlCard.style.transform = "scale(1)";
        htmlCard.style.opacity = "1";
      });

      currentScaleRef.current = 1;
    };

    const handleWheel = (e: WheelEvent) => {
      // Increase velocity detection for wheel events
      scrollVelocityRef.current += Math.abs(e.deltaY) * 0.5;
    };

    const handleTouchStart = () => {
      if (touchTimeoutRef.current) clearTimeout(touchTimeoutRef.current);
    };

    const handleTouchEnd = () => {
      touchTimeoutRef.current = setTimeout(() => {
        isScrollingRef.current = false;
        scrollVelocityRef.current = 0;
        bounceBackCards();
      }, 150);
    };

    // Initialize both animations
    build();
    ScrollTrigger.addEventListener("refreshInit", build);

    // Synchronized Scroll Event Listeners
    window.addEventListener("scroll", onScroll, { passive: true });
    window.addEventListener("wheel", handleWheel, { passive: true });
    document.addEventListener("touchstart", handleTouchStart, {
      passive: true,
    });
    document.addEventListener("touchend", handleTouchEnd, { passive: true });

    // Initialize scroll position
    updateScrollPosition();

    return () => {
      // GSAP Cleanup
      ScrollTrigger.removeEventListener("refreshInit", build);
      if (tlRef.current) {
        tlRef.current.scrollTrigger?.kill();
        tlRef.current.kill();
        tlRef.current = null;
      }

      // Synchronized Scroll Cleanup
      window.removeEventListener("scroll", onScroll);
      window.removeEventListener("wheel", handleWheel);
      document.removeEventListener("touchstart", handleTouchStart);
      document.removeEventListener("touchend", handleTouchEnd);

      // Clear timers
      if (scrollEndTimerRef.current) clearTimeout(scrollEndTimerRef.current);
      if (touchTimeoutRef.current) clearTimeout(touchTimeoutRef.current);
    };
  }, []);


 useEffect(() => {
    gsap.timeline({
      scrollTrigger: {
        trigger: ".review-wrap",
        start: "top top",
        end: "bottom+=6000 top", 
        scrub: 1.2,
      },
    })
      .to(".textbgclip", {
        clipPath: "inset(0px 0px 0px 0%)",
        ease: "power2.out",
      })
        // Hold phase
      .to(".textbgclip", {
        clipPath: "inset(0px 0px 0px 0%)",
        duration: 1,
        ease: "none",
      })
      .to(".textbgclip", {
        clipPath: "inset(0% 100% 0% 0%)",
        ease: "power2.inOut",
      });
  }, []);
  return (
    <div className="review-wrap" ref={containerRef}>
      <h2 className="textbgclip"  style={{ clipPath: "inset(0px 0px 0px 100%)"}}>TESTIMONIALS</h2>
      <h2 className="textbg">TESTIMONIALS</h2>
      <ul className="review-cards" ref={itemsRef}>
        <li>
          <h4 className="title">Noah Vertefeuille</h4>
          <p>
            Working with Moiz was a dream from start to finish. He&apos;s fast,
            creative, reliable, and incredibly easy to work with. Every design
            he delivered was not only beautiful but also strategically aligned
            with our goals and brand. He takes feedback gracefully, communicates
            clearly, and consistently goes above and beyond to deliver top-notch
            work
          </p>
        </li>
        <li>
          <h4 className="title">John Oberg</h4>
          <p>
            Moiz was incredible to work with and provided fast, professional
            work for our project. We will continue to return to Moiz whenever we
            have work for him! I can&apos;t recommend him enough!
          </p>
        </li>
        <li>
          <h4 className="title">Jovan Obradovic</h4>
          <p>
            Very punctual and professional. Easy to communicate and available at
            all times. Highly recommend
          </p>
        </li>
        <li>
          <h4 className="title">Alejandro Nunez</h4>
          <p>
            This is my second time collaborating with Moiz, and I had a great
            experience. He&apos;s very professional and responsive, and I look
            forward to collaborating again on new projects
          </p>
        </li>
        <li>
          <h4 className="title">Amari Agency</h4>
          <p>
            Moiz was really great to work with, very meticulous, attention to
            details, he got me what I wanted and was always available, was good
            with communications in regards to revisions. Will certainly work
            with him again.
          </p>
        </li>
        <li>
          <h4 className="title">Ryan Hurley</h4>
          <p>
            My second time working with Moiz for a figma design and it went
            great as always. Super fast, willing to make the changes and overall
            looks great. Could do a but better following instructions at first,
            but is happy to make the changes after providing examples. I&apos;ll
            be back.
          </p>
        </li>
        <li>
          <h4 className="title">Kayode Olusegun</h4>
          <p>
            Moiz ensures he understands your specifications, keeps you fully
            updated and communicated effectively. His job is very professional ,
            high standard and always according to the requirements
          </p>
        </li>
      </ul>
    </div>
  );
}
