import { useEffect, useRef } from "react";
import { Swiper } from "swiper";
import { Navigation } from "swiper/modules";
import Image from "react-bootstrap/Image";
import { gsap, ScrollTrigger } from "@/helpers/gsap";
import { Back } from "gsap";
import { Col, Row, Container, Tab, Nav } from "react-bootstrap";
import FlipButton from "../FlipButton";

gsap.registerPlugin(Back);

export default function CaseSwiperCursor() {
  /* ===================== REFS ===================== */
  const cursorRef = useRef<HTMLDivElement | null>(null);
  const cursorIconRef = useRef<HTMLImageElement  | null>(null);

  const clientX = useRef<number>(0);
  const clientY = useRef<number>(0);

  const cursorSide = useRef<"left" | "right" | null>(null);
  const cursorInside = useRef<boolean>(false);

  const easing = Back.easeOut.config(1.7);
  const animationDuration = 0.3;

  /* ===================== EFFECT ===================== */
  useEffect(() => {
    const cursor = cursorRef.current;
    const cursorIcon = cursorIconRef.current;
    const swiperContainer =
      document.querySelector<HTMLDivElement>(".swiper-container");
    const prevBtn = document.querySelector<HTMLDivElement>(
      ".swiper-button-prev"
    );
    const nextBtn = document.querySelector<HTMLDivElement>(
      ".swiper-button-next"
    );

    if (!cursor || !cursorIcon || !swiperContainer) return;

    /* ---------- Initial Cursor State ---------- */
    gsap.set(cursorIcon, {
      rotation: -135,
      opacity: 0,
      scale: 0.5,
    });
    gsap.set(cursor, {
      opacity: 0,
      scale: 1,
    });

    /* ---------- Mouse Move ---------- */
    const onMouseMove = (e: MouseEvent) => {
      clientX.current = e.clientX;
      clientY.current = e.clientY;
    };
    document.addEventListener("mousemove", onMouseMove);

    /* ---------- Render Loop ---------- */
    const render = () => {
      gsap.set(cursor, {
        x: clientX.current,
        y: clientY.current,
      });
      requestAnimationFrame(render);
    };
    render();

    /* ---------- Swiper Hover ---------- */
    const onEnter = (side: "left" | "right") => {
      cursorSide.current = side;

      gsap.to(cursorIcon, {
        rotation: side === "right" ? 0 : -180,
        scale: 1,
        opacity: 1,
        duration: animationDuration,
        ease: easing,
      });

      cursorInside.current = true;
    };

    const onLeave = () => {
      gsap.to(cursorIcon, {
        opacity: 0,
        scale: 0.3,
        duration: animationDuration,
      });

      cursorSide.current = null;
      cursorInside.current = false;
    };

    const switchSides = () => {
      if (!cursorInside.current || !cursorSide.current) return;

      gsap.to(cursorIcon, {
        rotation: cursorSide.current === "right" ? -180 : 0,
        duration: animationDuration,
        ease: easing,
      });

      cursorSide.current = cursorSide.current === "left" ? "right" : "left";
    };

    // swiperContainer.addEventListener("mouseenter", onEnter);
    // swiperContainer.addEventListener("mouseleave", onLeave);
    prevBtn?.addEventListener("mouseenter", () => {
      cursorSide.current = "left";

      gsap.to(cursor, {
        opacity: 1,
        duration: 0.15,
      });

      gsap.to(cursorIcon, {
        opacity: 1,
        scale: 1,
        rotation: -180,
        duration: animationDuration,
        ease: easing,
      });
    });
    nextBtn?.addEventListener("mouseenter", () => {
      cursorSide.current = "right";

      gsap.to(cursor, {
        opacity: 1,
        duration: 0.15,
      });

      gsap.to(cursorIcon, {
        opacity: 1,
        scale: 1,
        rotation: 0,
        duration: animationDuration,
        ease: easing,
      });
    });
    const hideCursor = () => {
      cursorSide.current = null;

      gsap.to(cursorIcon, {
        opacity: 0,
        scale: 0.4,
        duration: 0.2,
      });

      gsap.to(cursor, {
        opacity: 0,
        duration: 0.2,
      });
    };
    prevBtn?.addEventListener("mouseleave", hideCursor);
    nextBtn?.addEventListener("mouseleave", hideCursor);

    /* ---------- Swiper Init ---------- */
    const swiper = new Swiper(swiperContainer, {
      modules: [Navigation],
      loop: true,
      slidesPerView: "auto",
      // spaceBetween: 40,
      centeredSlides: true,
      speed: 800,
      navigation: {
        nextEl: ".swiper-button-next",
        prevEl: ".swiper-button-prev",
      },
    });

    /* ---------- Cursor Bump ---------- */
    const bump = gsap.to(cursor, {
      scale: 0.85,
      duration: 0.1,
      paused: true,
      onComplete: () => {
        gsap.to(cursor, {
          scale: 1,
          duration: 0.2,
          ease: easing,
        });
      },
    });

    swiper.on("slideChange", () => bump.play());

    /* ---------- Cleanup ---------- */
    return () => {
      document.removeEventListener("mousemove", onMouseMove);
      swiper.destroy(true, true);
    };
  }, []);

  /* ===================== JSX ===================== */
  return (
    <>
      {/* Cursor */}
      <div className="arrow-cursor" ref={cursorRef}>
        <Image src={'/images/arrow-right.svg'} alt="" ref={cursorIconRef} />
        {/* <svg
          ref={cursorIconRef}
          className="arrow-cursor__icon"
          viewBox="0 0 60 40"
        >
          <path className="arrow-cursor__path" d="M5 20h50M40 5l15 15-15 15" />
        </svg> */}
      </div>

      {/* Swiper */}
      <div className="swiper-container">
        <div className="swiper-wrapper">
          <div className="swiper-slide swiper-slide--wide case-study-card">
            <Image
              className="case-study__bg"
              src={`/images/group2147255963_rilnvg.webp`}
              alt=""
            />
            <div className="content">
              <div className="case-study-content">
                <div>
                  <h3 className="title">VELA SAIL WEBSITE</h3>
                  <p className="para">
                    The new Exit Ramp platform transformed a static MVP into a
                    fully operational M&A marketplace — scalable, secure, and
                    user-centric.
                  </p>
                </div>
                <FlipButton text="Case Study" className="btn-green" href="/case-studies/vela-sail" />
              </div>
              <Image
                className="case-study__img"
                src={`/images/frame2147257383_bfhcwx.webp`}
                alt=""
              />
            </div>
          </div>
          <div className="swiper-slide swiper-slide--wide case-study-card">
            <Image
              className="case-study__bg"
              src={`/images/rectangle346271121_ogtfno.webp`}
              alt=""
            />
            <div className="content">
              <div className="case-study-content">
                <div>
                  <h3 className="title">My Exit Ramp Dashboard</h3>
                  <p className="para">
                    The new Exit Ramp platform transformed a static MVP into a
                    fully operational M&A marketplace — scalable, secure, and
                    user-centric.
                  </p>
                </div>
                <FlipButton text="Case Study" className="btn-green" href="/case-studies/my-exit-ramp" />
              </div>
              <Image
                className="case-study__img"
                src={`/images/group2147255964_vcmk6x.webp`}
                alt=""
              />
            </div>
          </div>
          <div className="swiper-slide swiper-slide--wide case-study-card">
            <Image
              className="case-study__bg"
              src={`/images/group2147255966_ae5uqg.webp`}
              alt=""
            />
            <div className="content">
              <div className="case-study-content">
                <div>
                  <h3 className="title">NICKL Pass Mobile App</h3>
                  <p className="para">
                    The new Exit Ramp platform transformed a static MVP into a
                    fully operational M&A marketplace — scalable, secure, and
                    user-centric.
                  </p>
                </div>
                <FlipButton text="Case Study" className="btn-green" href="/case-studies/nickl-pass" />
              </div>
              <Image
                className="case-study__img"
                src={`/images/group2147255965_huc3vb.webp`}
                alt=""
              />
            </div>
          </div>
          <div className="swiper-slide swiper-slide--wide case-study-card">
            <Image
              className="case-study__bg"
              src={`/images/group2147255965_1_tctjqk.webp`}
              alt=""
            />
            <div className="content">
              <div className="case-study-content">
                <div>
                  <h3 className="title">Paws Contest Mobile App</h3>
                  <p className="para">
                    The new Exit Ramp platform transformed a static MVP into a
                    fully operational M&A marketplace — scalable, secure, and
                    user-centric.
                  </p>
                </div>
                <FlipButton text="Case Study" className="btn-green" href="#" />
              </div>
              <Image
                className="case-study__img"
                src={`/images/frame2147257384_msf55k.webp`}
                alt=""
              />
            </div>
          </div>
        
        </div>

        <div className="swiper-button-prev" />
        <div className="swiper-button-next" />
      </div>
    </>
  );
}
