"use client";

import { useEffect, useRef } from "react";
import { Swiper } from "swiper";
import { Navigation } from "swiper/modules";
import Image from "react-bootstrap/Image";
import { gsap, Back } from "gsap";
import FlipButton from "../FlipButton";

gsap.registerPlugin(Back);

export default function StuffSwiperCursor() {
  /* ===================== REFS ===================== */
  const swiperRef = useRef<HTMLDivElement | null>(null);
  const prevRef = useRef<HTMLDivElement | null>(null);
  const nextRef = useRef<HTMLDivElement | null>(null);

  const cursorRef = useRef<HTMLDivElement | null>(null);
  const cursorIconRef = useRef<HTMLImageElement | null>(null);

  const mouseX = useRef(0);
  const mouseY = useRef(0);

  const easing = Back.easeOut.config(1.7);

  /* ===================== EFFECT ===================== */
  useEffect(() => {
    const swiperEl = swiperRef.current;
    const prevBtn = prevRef.current;
    const nextBtn = nextRef.current;
    const cursor = cursorRef.current;
    const cursorIcon = cursorIconRef.current;

    if (!swiperEl || !prevBtn || !nextBtn || !cursor || !cursorIcon) return;

    /* ---------- Cursor Initial ---------- */
    gsap.set(cursor, { opacity: 0 });
    gsap.set(cursorIcon, { opacity: 0, scale: 0.5, rotation: -135 });

    /* ---------- Mouse Tracking ---------- */
    const onMouseMove = (e: MouseEvent) => {
      mouseX.current = e.clientX;
      mouseY.current = e.clientY;
    };
    window.addEventListener("mousemove", onMouseMove);

    gsap.ticker.add(() => {
      gsap.set(cursor, {
        x: mouseX.current,
        y: mouseY.current,
      });
    });

    /* ---------- Hover Handlers ---------- */
    const showLeft = () => {
      gsap.to(cursor, { opacity: 1, duration: 0.15 });
      gsap.to(cursorIcon, {
        opacity: 1,
        scale: 1,
        rotation: -180,
        duration: 0.3,
        ease: easing,
      });
    };

    const showRight = () => {
      gsap.to(cursor, { opacity: 1, duration: 0.15 });
      gsap.to(cursorIcon, {
        opacity: 1,
        scale: 1,
        rotation: 0,
        duration: 0.3,
        ease: easing,
      });
    };

    const hideCursor = () => {
      gsap.to(cursor, { opacity: 0, duration: 0.2 });
      gsap.to(cursorIcon, {
        opacity: 0,
        scale: 0.4,
        duration: 0.2,
      });
    };

    prevBtn.addEventListener("mouseenter", showLeft);
    nextBtn.addEventListener("mouseenter", showRight);
    prevBtn.addEventListener("mouseleave", hideCursor);
    nextBtn.addEventListener("mouseleave", hideCursor);

    /* ---------- Swiper Init ---------- */
    const swiper = new Swiper(swiperEl, {
      modules: [Navigation],
      loop: true,
      centeredSlides: true,
      slidesPerView: "auto",
      speed: 800,
      navigation: {
        prevEl: prevBtn,
        nextEl: nextBtn,
      },
    });

    /* ---------- Cursor Bump on Slide ---------- */
    swiper.on("slideChange", () => {
      gsap.fromTo(
        cursor,
        { scale: 0.85 },
        { scale: 1, duration: 0.25, ease: easing }
      );
    });

    /* ---------- Cleanup ---------- */
    return () => {
      window.removeEventListener("mousemove", onMouseMove);
      prevBtn.removeEventListener("mouseenter", showLeft);
      nextBtn.removeEventListener("mouseenter", showRight);
      prevBtn.removeEventListener("mouseleave", hideCursor);
      nextBtn.removeEventListener("mouseleave", hideCursor);
      swiper.destroy(true, true);
    };
  }, []);

  /* ===================== JSX ===================== */
  return (
    <>
      {/* Cursor */}
      <div className="arrow-cursor" ref={cursorRef}>
        <Image
          src="/images/arrow-right.svg"
          alt=""
          ref={cursorIconRef}
        />
      </div>

      {/* Swiper */}
      <div className="swiper-container" ref={swiperRef}>
        <div className="swiper-wrapper">
            <div className="swiper-slide swiper-slide--wide case-study-card">
              <Image
                className="case-study__bg"
                src={`/images/group1686569693_ydke5y.webp`}
                alt=""
              />
            </div>
            <div className="swiper-slide swiper-slide--wide case-study-card">
              <Image
                className="case-study__bg"
                src={`/images/group2147255935_q9jb8v.webp`}
                alt=""
              />
            </div>
            <div className="swiper-slide swiper-slide--wide case-study-card">
              <Image
                className="case-study__bg"
                src={`/images/group1686569692_xbpfnx.webp`}
                alt=""
              />
            </div>
            <div className="swiper-slide swiper-slide--wide case-study-card">
              <Image
                className="case-study__bg"
                src={`/images/group2147255937_patiem.webp`}
                alt=""
              />
            </div>
        </div>

        <div className="swiper-button-prev" ref={prevRef} />
        <div className="swiper-button-next" ref={nextRef} />
      </div>
    </>
  );
}
